<?php
use Elementor\Widget_Base;
use Elementor\Controls_Manager;

if ( ! defined( 'ABSPATH' ) ) exit;

class ACS_Carousel_Widget extends Widget_Base {

    public function get_name() {
        return 'acs_carousel';
    }

    public function get_title() {
        return __( 'Advanced Carousel Scroll', 'acs-carousel' );
    }

    public function get_icon() {
        return 'eicon-slider-push';
    }

    public function get_categories() {
        return [ 'general' ];
    }

    protected function _register_controls() {
        $this->start_controls_section(
            'section_slides',
            [ 'label' => __( 'Slides', 'acs-carousel' ) ]
        );

        $repeater = new \Elementor\Repeater();

        $repeater->add_control('image', [
            'label' => __('Image', 'acs-carousel'),
            'type' => Controls_Manager::MEDIA
        ]);

        $repeater->add_control('title', [
            'label' => __('Title', 'acs-carousel'),
            'type' => Controls_Manager::TEXT,
            'default' => 'Slide Title'
        ]);

        $repeater->add_control('text', [
            'label' => __('Text', 'acs-carousel'),
            'type' => Controls_Manager::TEXTAREA,
            'default' => 'Slide description...'
        ]);

        $repeater->add_control('link', [
            'label' => __('Link', 'acs-carousel'),
            'type' => Controls_Manager::URL,
            'placeholder' => 'https://your-link.com',
            'default' => []
        ]);

        $this->add_control('slides', [
            'label' => __('Slides', 'acs-carousel'),
            'type' => Controls_Manager::REPEATER,
            'fields' => $repeater->get_controls(),
            'title_field' => '{{{ title }}}'
        ]);

        $this->end_controls_section();

        $this->start_controls_section(
            'carousel_settings',
            ['label' => __('Carousel Settings', 'acs-carousel')]
        );

        $this->add_control('autoplay_speed', [
            'label' => __('Autoplay Speed (ms)', 'acs-carousel'),
            'type' => Controls_Manager::NUMBER,
            'default' => 3000
        ]);

        $this->add_control('slides_to_show', [
            'label' => __('Slides to Show', 'acs-carousel'),
            'type' => Controls_Manager::SELECT,
            'options' => ['1'=>'1', '2'=>'2', '3'=>'3', '4'=>'4'],
            'default' => '1'
        ]);

        $this->end_controls_section();
    }

    protected function render() {
        $settings = $this->get_settings_for_display();
        $autoplay = intval($settings['autoplay_speed']);
        $slidesToShow = intval($settings['slides_to_show']);
        $uid = uniqid("acs_");

        ?>
        <style>
            #<?php echo $uid; ?> {
                position: relative;
                overflow: hidden;
            }
            #<?php echo $uid; ?> .acs-carousel-track {
                display: flex;
                transition: transform 0.5s ease-in-out;
                will-change: transform;
            }
            #<?php echo $uid; ?> .acs-slide {
                flex: 0 0 calc(100% / <?php echo esc_attr($slidesToShow); ?>);
                box-sizing: border-box;
                padding: 10px;
                text-align: center;
            }
            #<?php echo $uid; ?> .acs-slide img {
                max-width: 100%;
                height: auto;
                display: block;
                margin: 0 auto 10px;
            }
            #<?php echo $uid; ?> .acs-nav {
                position: absolute;
                top: 50%;
                transform: translateY(-50%);
                background: rgba(0,0,0,0.5);
                color: #fff;
                border: none;
                padding: 10px;
                cursor: pointer;
                z-index: 10;
            }
            #<?php echo $uid; ?> .acs-nav.prev {
                left: 0;
            }
            #<?php echo $uid; ?> .acs-nav.next {
                right: 0;
            }
        </style>

        <div id="<?php echo $uid; ?>" class="acs-carousel-wrapper" data-autoplay="<?php echo $autoplay; ?>" data-show="<?php echo $slidesToShow; ?>">
            <button class="acs-nav prev">&#10094;</button>
            <div class="acs-carousel-track">
                <?php foreach ($settings['slides'] as $slide): ?>
                    <div class="acs-slide">
                        <?php if (!empty($slide['link']['url'])): ?>
                        <a href="<?php echo esc_url($slide['link']['url']); ?>" <?php echo $slide['link']['is_external'] ? 'target="_blank"' : ''; ?>>
                        <?php endif; ?>
                            <?php if ($slide['image']['url']): ?>
                                <img src="<?php echo esc_url($slide['image']['url']); ?>" alt="<?php echo esc_attr($slide['title']); ?>">
                            <?php endif; ?>
                            <h4><?php echo esc_html($slide['title']); ?></h4>
                            <p><?php echo esc_html($slide['text']); ?></p>
                        <?php if (!empty($slide['link']['url'])): ?>
                        </a>
                        <?php endif; ?>
                    </div>
                <?php endforeach; ?>
            </div>
            <button class="acs-nav next">&#10095;</button>
        </div>

        <script>
        (function(){
            const el = document.getElementById('<?php echo $uid; ?>');
            const track = el.querySelector('.acs-carousel-track');
            const slides = el.querySelectorAll('.acs-slide');
            const prevBtn = el.querySelector('.acs-nav.prev');
            const nextBtn = el.querySelector('.acs-nav.next');
            const autoplay = parseInt(el.getAttribute('data-autoplay'));
            const show = parseInt(el.getAttribute('data-show'));
            let current = 0;

            function update() {
                const slideWidth = slides[0].offsetWidth;
                track.style.transform = 'translateX(' + (-current * slideWidth) + 'px)';
            }

            prevBtn.addEventListener('click', () => {
                if (current > 0) current--;
                update();
            });

            nextBtn.addEventListener('click', () => {
                if (current < slides.length - show) current++;
                update();
            });

            if (autoplay > 0) {
                setInterval(() => {
                    current = (current + 1) % (slides.length - show + 1);
                    update();
                }, autoplay);
            }

            window.addEventListener('resize', update);
        })();
        </script>
        <?php
    }
}
